#include "shellmanager.h"
#include <QProgressDialog>
#include <QMutex>
#include <QFileInfo>
#include <QWaitCondition>
#include <QMessageBox>
#include <QApplication>



ShellManager::ShellManager(QObject *parent)
	: QObject(parent)
{

	//initialize command shell process spawner
	m_CmdShellEnv.insert("HOME","ffpresets");
	m_CmdShellProcess.setWorkingDirectory(QCoreApplication::applicationDirPath());
	m_CmdShellProcess.setProcessEnvironment (m_CmdShellEnv);
	connect(&m_CmdShellProcess,SIGNAL(finished(int,QProcess::ExitStatus)),this,SLOT(OnCommandShellProcessFinished(int,QProcess::ExitStatus)));
	connect(&m_CmdShellProcess,SIGNAL(error(QProcess::ProcessError)),this,SLOT(OnCommandShellProcessError(QProcess::ProcessError)));

}

ShellManager::~ShellManager()
{

}



bool ShellManager::ExecuteShellCommand(QString strShellCommand, QStringList lstArgs, bool nIgnoreExitCode)
{
	QProcessEnvironment env;
	env.insert("HOME","ffpresets");
	QProcess app;
	app.setWorkingDirectory(QCoreApplication::applicationDirPath());
	app.setProcessEnvironment (env);
	app.start(strShellCommand,lstArgs);
	if (!app.waitForFinished(-1)) //wait forever
	{		
		m_byteCmdShellStdErrOutput=app.readAllStandardError();
		QByteArray errOut=app.readAllStandardOutput();
		if (m_byteCmdShellStdErrOutput.size()>0)
			QMessageBox::information(NULL,"Information","Operation failed!"+m_byteCmdShellStdErrOutput.left(4095));
		else
			QMessageBox::information(NULL,"Information","Operation failed!"+errOut.left(4095));
		return false;
	}

	int nExitCode=app.exitCode();
	m_byteCmdShellStdErrOutput=app.readAllStandardError();
	QByteArray errOut=app.readAllStandardOutput();

	if (nIgnoreExitCode)
		return true;

	if (nExitCode!=0)
	{
		if (m_byteCmdShellStdErrOutput.size()>0)
			QMessageBox::information(NULL,"Information","Operation failed!"+m_byteCmdShellStdErrOutput.left(4095));
		else
			QMessageBox::information(NULL,"Information","Operation failed!"+errOut.left(4095));

		return false;
	}

	return true;
}
//start command in the shell async, return to program immediately 
void ShellManager::StartShellCommand(QString strShellCommand, QStringList lstArgs)
{
	//clear error status before spawning new cmd shell process
	m_strLastErrorCmdShell="";
	m_bErrorOccuredCmdShell=false;
	m_strCmdShellApplicationName=strShellCommand;

	m_CmdShellProcess.start(strShellCommand,lstArgs);
}


void ShellManager::OnCommandShellProcessFinished(int errCode ,QProcess::ExitStatus status)
{
	if (status != QProcess::NormalExit)
	{
		m_bErrorOccuredCmdShell=true;
		m_strLastErrorCmdShell="Application failed to execute!";
		return;
	}
	else
	{
		if (errCode!=0)
		{
			QByteArray errContent=m_CmdShellProcess.readAllStandardError();
			m_byteCmdShellStdErrOutput=errContent;
			QByteArray errOut=m_CmdShellProcess.readAllStandardOutput();
			if (errContent.size()>0)
				m_strLastErrorCmdShell=errContent.left(4095);
			else
				m_strLastErrorCmdShell=errOut.left(4095);

			if (m_strLastErrorCmdShell.isEmpty())
				m_strLastErrorCmdShell="Application failed to execute!";

			m_bErrorOccuredCmdShell=true;
		}
	}
}


void ShellManager::OnCommandShellProcessError(QProcess::ProcessError error)
{
	if (error==QProcess::FailedToStart)
	{
		m_strLastErrorCmdShell=QString("Application %1 failed to start. Please make sure that application exist in path!").arg(m_strCmdShellApplicationName);
		m_bErrorOccuredCmdShell=true;
		return;
	}

	QByteArray errContent=m_CmdShellProcess.readAllStandardError();
	m_byteCmdShellStdErrOutput=errContent;
	QByteArray errOut=m_CmdShellProcess.readAllStandardOutput();
	if (errContent.size()>0)
		m_strLastErrorCmdShell=errContent.left(4095);
	else
		m_strLastErrorCmdShell=errOut.left(4095);

	if (m_strLastErrorCmdShell.isEmpty())
		m_strLastErrorCmdShell="Application failed to execute!";

	m_bErrorOccuredCmdShell=true;
}


//checks status of cmd shell process, do not exits until cmd shell is finsihed
//shows progress bar, checks file size if available, based on current file size and target file size calculate progress bar (approx)
//if error occurred, show message and return false, else true
//bIgnoreErrorIfProgressActivated = some programs always return 1 as exit code: ignore it if resulting file is created.
bool ShellManager::WaitForCmdShellToFinish(QString strFilePathToCheckSize, int nTargetSize, QString strProgressBarMsg, bool bIgnoreErrorIfProgressActivated)
{
	int nCurrentFileSize=0;
	QSize size(300,60);
	QProgressDialog progress(dynamic_cast<QWidget*>(this->parent())); //parent is GUI window, get his pointer as widget to center progress and make it modal
	progress.setLabelText(strProgressBarMsg);
	progress.setWindowTitle("Processing video file");
	progress.setCancelButton(0);
	progress.setMinimumSize(size);
	progress.setWindowFlags(Qt::Window | Qt::CustomizeWindowHint | Qt::WindowTitleHint| Qt::WindowSystemMenuHint); 
	progress.setRange(1,100);
	progress.setValue(2);
	progress.setModal(true);

	int nValue = 0;
	
	while(m_CmdShellProcess.state() != QProcess::NotRunning)
	{
		if (!strFilePathToCheckSize.isEmpty() && nTargetSize!=0)
		{
			QFileInfo info_curr(strFilePathToCheckSize);
			nCurrentFileSize = info_curr.size();
			nValue=((float)nCurrentFileSize/nTargetSize)*100;
		}

		if (nValue>=100) //if get to end means, estimate is wrong: increase targetsize by 20%
		{
			nValue=90;
			nTargetSize+=nTargetSize*0.2;
		}
		progress.setValue(nValue);
		QCoreApplication::processEvents(QEventLoop::ExcludeUserInputEvents); //refresh GUI

		//go sleep 1 sec:
		Sleep(1000);
	}

	//process over: check status:
	if (m_bErrorOccuredCmdShell && (!bIgnoreErrorIfProgressActivated || nValue==0) )
	{
		QMessageBox::information(NULL,"Information","Operation failed: "+m_strLastErrorCmdShell);
		return false;
	}
	else
		return true;

}

void ShellManager::Sleep(unsigned long miliseconds)
{
	QMutex mutex;
	mutex.lock();
	QWaitCondition holdthread;
	holdthread.wait(&mutex,miliseconds);
	mutex.unlock();
}
