#include "mergemp4.h"
#include "shellmanager.h"
#include "videohelper.h"
#include <QFileDialog>
#include <QMessageBox>
#include <QUrl>
#include <QDesktopServices>




#define TMP_DIR_URL		"<html><body style=\" font-family:'Arial'; font-size:8pt; font-weight:500; font-style:normal;\"><p style=\" margin-top:0px; margin-bottom:0px; margin-left:0px; margin-right:0px; -qt-block-indent:0; text-indent:0px;\"><a href=\"%1\"><span style=\" text-decoration: underline; color:#0000ff;\">%2</span></a></p></body></html>"
#define STYLE_BUTTON 	"QPushButton { border-width: 4px; color: white;border-image:url(%1) 4px 4px 4px 4px stretch stretch } QPushButton:hover {  border-width: 4px; color: black;border-image:url(%2) 4px 4px 4px 4px stretch stretch}"

ShellManager *g_ShellManager = NULL; //use shell manager as global object

MergeMp4::MergeMp4(QWidget *parent, Qt::WFlags flags)
	: QMainWindow(parent, flags)
{
	ui.setupUi(this);
	m_strPathOut="C:/out.mp4";
	ui.lineEdit->setText(m_strPathOut);
	setWindowTitle("Merge and Re-encode video files to mp4 format - Version 3.2");

	//setup table widget
	connect(ui.tableWidget,SIGNAL(SignalFilesDropped(QList<QUrl>)),this,SLOT(OnSignalFilesDropped(QList<QUrl>)));
	connect(ui.tableWidget,SIGNAL(itemSelectionChanged()),this,SLOT(OnitemSelectionChanged()));

	//enable link clickable
	ui.labelPath->setText(QString(TMP_DIR_URL).arg("file://"+QDir::tempPath()).arg(QDir::tempPath()));
	connect(ui.labelPath,SIGNAL(linkActivated ( const QString &)),this,SLOT(OnPathLinkActivated ( const QString &)));

	
	m_ProgresBar = new QProgressBar();
	m_ProgresBar->setMinimumWidth(200);
	m_ProgresBar->setMaximumWidth(200);
	m_ProgresBar->setMaximumHeight(16);
	m_ProgresBar->setRange(1,100);
	this->statusBar()->addPermanentWidget(m_ProgresBar);

	this->statusBar()->clearMessage();
	m_ProgresBar->setVisible(false);


	g_ShellManager = new ShellManager(this); //instancing with parent=this, ensures automatic deletion of the object as child objects are deleted along with parent

	//set background:
	this->centralWidget()->setObjectName("TARGET_STYLE");
	QString strStyle = "QWidget#%1 { background:qradialgradient(cx:0.8, cy:0.3, radius: 1, fx:0.8 fy:0.5, stop:0 rgb(151,151,151), stop:1 rgb(100,100,100));} QLabel {color:black}";
	this->setStyleSheet(strStyle.arg("TARGET_STYLE"));

	QString strStyle1 = "QWidget#%1 { background:qlineargradient(x1:0, y1:0, x2:0, y2:0.5, stop:0 rgb(57,57,57), stop:1  rgb(111,111,111));} QLabel {color:white}";
	ui.statusBar->setStyleSheet(strStyle1.arg("statusBar"));

	ui.btnAdd->setStyleSheet(QString(STYLE_BUTTON).arg(":GreyBtn.png").arg(":GreyBtn_Hover.png"));
	ui.btnDelete->setStyleSheet(QString(STYLE_BUTTON).arg(":GreyBtn.png").arg(":GreyBtn_Hover.png"));
	ui.btnClear->setStyleSheet(QString(STYLE_BUTTON).arg(":GreyBtn.png").arg(":GreyBtn_Hover.png"));
	ui.btnUp->setStyleSheet(QString(STYLE_BUTTON).arg(":GreyBtn.png").arg(":GreyBtn_Hover.png"));
	ui.btnDown->setStyleSheet(QString(STYLE_BUTTON).arg(":GreyBtn.png").arg(":GreyBtn_Hover.png"));
	ui.btnBrowse->setStyleSheet(QString(STYLE_BUTTON).arg(":GreyBtn.png").arg(":GreyBtn_Hover.png"));
	ui.btnMerge->setStyleSheet(QString(STYLE_BUTTON).arg(":GreyBtn.png").arg(":GreyBtn_Hover.png"));

	
}

MergeMp4::~MergeMp4()
{

}

void MergeMp4::on_btnClear_clicked()
{
	ui.tableWidget->DeleteAll();
	ui.lineEdit->clear();
}

void MergeMp4::OnitemSelectionChanged()
{
	int nRow=ui.tableWidget->currentRow();
	QString strFile=ui.tableWidget->item(nRow,0)->text(); //take filename
	QFileInfo info(strFile);
	QFileInfo info2(ui.lineEdit->text());
	strFile=info.absolutePath()+"/"+info.baseName()+".mp4";

	if (info2.fileName()=="out.mp4" || ui.lineEdit->text().isEmpty())
	{
		ui.lineEdit->setText(QDir::cleanPath(strFile));
	}
}


void MergeMp4::OnSignalFilesDropped(QList<QUrl> lstFilePaths)
{
	QStringList lstFiles;

	for (int i =0;i<lstFilePaths.size();i++)
	{
		QFileInfo info(lstFilePaths.at(i).toLocalFile());
		//if (info.suffix().toLower()=="mp4" || info.suffix().toLower()=="flv" || info.suffix().toLower()=="mpg" || info.suffix().toLower()=="mpeg" || info.suffix().toLower()=="avi")
		//{
		lstFiles<<lstFilePaths.at(i).toLocalFile();
		//}
	}

	DoAddFiles(lstFiles);
}

void MergeMp4::on_btnAdd_clicked()
{
	QStringList	lstFiles= QFileDialog::getOpenFileNames ( this, "Choose video files", m_strPath, "Video Files (*.mp4 *.mpg *.mpeg *.flv  *.avi)");
	DoAddFiles(lstFiles);
}


void MergeMp4::DoAddFiles(QStringList lstFiles)
{
	for (int i=0;i<lstFiles.size();i++)
	{
		QString strFile=lstFiles.at(i);

		//check if already same path added: if so then skip:
		for (int i =0;i<ui.tableWidget->rowCount();i++)
			if (strFile == ui.tableWidget->item(i,0)->text())
				continue;

		
		QString strSize;
		int nVideoBitrate;
		QString strAuxData;
		QString strFPS;
		QString strDAR;
		int nAudioBitRate;
		int nAudioFrequency;
		int nAudioChannels;
		QString strVideoCodec;
		QString strAudioCodec;

		if(!VideoHelper::GetSingleVideoFileInfo(strFile,strSize,nVideoBitrate,strAuxData,strFPS,strDAR,nAudioBitRate,nAudioFrequency,strVideoCodec,strAudioCodec,nAudioChannels))
		{
			QMessageBox::warning(this,"Error",QString("File %1 can not be analyzed! Probably unknown format or filename contain non ascii characters.").arg(strFile));
			return;
		}
		ui.tableWidget->AddFile(strFile,strVideoCodec,QString::number(nVideoBitrate),strSize,strAudioCodec,QString::number(nAudioFrequency),QString::number(nAudioChannels));
	}


	if (lstFiles.size()>0)
	{
		QFileInfo info (lstFiles.at(0));
		QString strFile=info.absolutePath()+"/"+info.baseName()+"_out.mp4";
		ui.lineEdit->setText(QDir::cleanPath(strFile));
		m_strPath=info.absolutePath();
	}

}


void MergeMp4::on_btnDelete_clicked()
{
	ui.tableWidget->DeleteSelected();
}

void MergeMp4::on_btnUp_clicked()
{
	ui.tableWidget->MoveUpSelected();
}

void MergeMp4::on_btnDown_clicked()
{
	ui.tableWidget->MoveDownSelected();
}

void MergeMp4::on_btnBrowse_clicked()
{
	QString strFil="out.mp4";
	QString strFile= QFileDialog::getSaveFileName ( this, "Save output video file", m_strPath, "Video File (*.mp4)",&strFil);
	if (!strFile.isEmpty())
	{
		ui.lineEdit->setText(QDir::cleanPath(strFile));
		QFileInfo info (strFile);
		m_strPath=info.absolutePath(); //just store for next time to open file dialog to right dir
	}

}


void MergeMp4::on_btnMerge_clicked()
{
	this->statusBar()->clearMessage();
	m_ProgresBar->setValue(1);
	m_ProgresBar->setVisible(true);
	QCoreApplication::processEvents(QEventLoop::ExcludeUserInputEvents);

	DoMerge();

	this->statusBar()->clearMessage();
	m_ProgresBar->setVisible(false);
}

void MergeMp4::DoMerge()
{
	//check output file
	if (QFile::exists(ui.lineEdit->text()))
	{
		int nRes=QMessageBox::question(this,"Warning","File "+ui.lineEdit->text()+ " already exists. Overwrite?","Yes","No");
		if (nRes!=0)
		{
			return;
		}
		QFile::remove(ui.lineEdit->text()); //remove manually..
	}

	//assure that output is mp4
	QFileInfo info_out(ui.lineEdit->text());
	if (info_out.suffix().toLower()!="mp4")
	{
		ui.lineEdit->setText(ui.lineEdit->text()+".mp4");
	}

	//get file paths from grid:
	QStringList lstFilePaths;
	for (int i =0;i<ui.tableWidget->rowCount();i++)
		lstFilePaths << ui.tableWidget->item(i,0)->text();

	
	//-------------------------------------------------------------------------------------------------------------------------------
	//Analyze all files, find best and fastest method to merge, optimally without re-encoding
	//-------------------------------------------------------------------------------------------------------------------------------
	bool bNeed2Convert=false;
	QString strFrameSize;
	QString strFPS;
	QString strDAR;
	int nVideoBitRateMp4;
	int nAudioBitRateMp4;
	int nAudioBitRateMpg;
	int nAudioFrequency;
	QStringList lstVideoCodecs;

	//refresh status
	m_ProgresBar->setValue(5);
	this->statusBar()->showMessage("Analyzing files");
	QCoreApplication::processEvents(QEventLoop::ExcludeUserInputEvents);

	//analyze files
	if (!AnaylizeFiles(lstFilePaths, bNeed2Convert, strFrameSize,strFPS,strDAR,nVideoBitRateMp4,nAudioBitRateMp4,nAudioBitRateMpg,nAudioFrequency,lstVideoCodecs))
		return;

	//refresh status
	m_ProgresBar->setValue(16);
	QCoreApplication::processEvents(QEventLoop::ExcludeUserInputEvents);

	QStringList lstTempResizedToMp4Fast;


	//-------------------------------------------------------------------------------------------------------------------------------
	// FLV or other containers can contain H264 stream, converting to mp4 can be fast, test if possible -> can speed up merge process
	// No sense to this operation if files need to be re-encoded
	//-------------------------------------------------------------------------------------------------------------------------------
	if (!bNeed2Convert)
	{
		//refresh status
		this->statusBar()->showMessage("Extracting H264 stream from FLV files"); 
		QCoreApplication::processEvents(QEventLoop::ExcludeUserInputEvents);

		QStringList lstFilesOut;

		//if only one then this is target file
		if (lstFilePaths.size()==1)
		{
			QFileInfo info(lstFilePaths.at(0));
			if(info.suffix().toLower().right(3)!="mp4" && lstVideoCodecs.at(0).indexOf("264")>=0) //only if contains h264 stream and not already mp4
			{
				if (!VideoHelper::ConvertToMp4Fast(lstFilePaths.at(0), ui.lineEdit->text()))
					return;

				m_ProgresBar->setValue(100);
				QCoreApplication::processEvents(QEventLoop::ExcludeUserInputEvents);

				QMessageBox::information(this,"Information","Operation success!");
				return;
			}
			else
				bNeed2Convert=true;

		}
		else //if more then probably only some files can be converted to mp4, if all converted then merge will be superfast.
		{
			if (!VideoHelper::ConvertToMp4Fast_Batch(lstFilePaths, lstVideoCodecs, lstFilesOut, lstTempResizedToMp4Fast))
				return;

			lstFilePaths = lstFilesOut; //now source files are newly created files

			//test if all files are in mp4 format: if only one is not, then set flag for conversion:
			for (int i =0;i<lstFilePaths.size();i++)
			{
				QString strFileIn = lstFilePaths.at(i);
				QFileInfo info(strFileIn);
				if(info.suffix().toLower().right(3)!="mp4")
				{
					bNeed2Convert=true;
					break;
				}
			}
		}

	}


	//refresh status
	m_ProgresBar->setValue(20);
	QCoreApplication::processEvents(QEventLoop::ExcludeUserInputEvents);


	QStringList lstTempResized;
	bool bConvertedToMpeg=false;


	//-----------------------------------------------------------------------------------------
	// Re-encode all files if not all of the same size, fps, aspect ratio, audio or video codec
	//-----------------------------------------------------------------------------------------
	if (bNeed2Convert || ui.ckbRencode->isChecked())
	{
		if (lstFilePaths.size()>1)//only resize if more then 1 file
		{
			//ask before convert
			if (!ui.ckbRencode->isChecked() ) //ask to re-encode only if not already forced or only 1 file then re-encoding is default
			{
				int nRes=QMessageBox::question(this,"Warning","Some video files are in different format (different size, fps or video codec is not H264). Re-encode to the same size and merge?","Yes","No");
				if (nRes!=0)
				{
					return;
				}
			}

			//refresh status
			this->statusBar()->showMessage("Converting files to mpg");
			QCoreApplication::processEvents(QEventLoop::ExcludeUserInputEvents);


			QApplication::setOverrideCursor(Qt::WaitCursor);

			//----------------------------------------------------------------------------
			//NOTE: logical step is to re-encode both audio & video to mp4 then to simply merge with mkvmerge, but re-encoding kicks audio stream out of sync, 
			//		interestingly mpeg2 stream with mpeg2 video stream is always in sync: so convert to mpg, join mpg then re-encode back to mp4 as mpeg is 4-5times bigger then mp4
			//----------------------------------------------------------------------------

			/*
			//convert to mp4
			if(!VideoHelper::ConvertToMp4_Resize(lstFilePaths,lstTempResized,strFrameSize,strFPS,strDAR,nVideoBitRateMp4,nAudioBitRateMp4,nAudioFrequency))
			{
				QApplication::restoreOverrideCursor();
				return;
			}
			else
			{
				//bConvertedToMpeg=true;
				lstFilePaths=lstTempResized; //after convert, use temporary file paths as source: merge with DOS copy or mp4box
			}
			*/

			//convert to mpg
			if(!VideoHelper::ConvertToMpeg_Resize(lstFilePaths,lstTempResized,strFrameSize,strFPS,strDAR,nAudioBitRateMpg,nAudioFrequency))
			{
				QApplication::restoreOverrideCursor();
				return;
			}
			else
			{
				bConvertedToMpeg=true;
				lstFilePaths=lstTempResized; //after convert, use temporary file paths as source: merge with DOS copy or mp4box
			}

			QApplication::restoreOverrideCursor();

		}
		else
		{
			bConvertedToMpeg=true; //if only 1 file then just re-encode source to mp4
		}
	}

	//refresh status
	this->statusBar()->showMessage("Merging files");
	m_ProgresBar->setValue(70);
	QCoreApplication::processEvents(QEventLoop::ExcludeUserInputEvents);

	bool bSuccess=false;
	
	if (bConvertedToMpeg)
	{
		//---------------------------------------------------------------
		// MERGE MPG FILES WITH DOS COMMAND: copy /b f1+f2+f3+... target
		//---------------------------------------------------------------
		QString strFileIn="";
		QString strTempMergedFileOut="";

		if (lstFilePaths.size()>1)
		{
			QFileInfo info2(ui.lineEdit->text());
			strTempMergedFileOut=QDir::tempPath()+"/"+info2.baseName()+"_temp_out.mpg";
			QFile::remove(strTempMergedFileOut);

			if(!VideoHelper::MergeMpeg(lstFilePaths,strTempMergedFileOut))
			{
				QMessageBox::warning(this,"Error","Error while merging file");
			}

			strFileIn=strTempMergedFileOut;
		}
		else
			strFileIn = lstFilePaths.at(0); //if only 1 file then no need for merge just convert to the mp4

		this->statusBar()->showMessage("Re-encoding to mp4");
		m_ProgresBar->setValue(87);
		QCoreApplication::processEvents(QEventLoop::ExcludeUserInputEvents);

		//convert result to mp4
		bSuccess = VideoHelper::ConvertToMp4(strFileIn,ui.lineEdit->text(),nVideoBitRateMp4,nAudioBitRateMp4,nAudioFrequency,false /*ui.ckbAudio->isChecked()*/);

		//delete temporary:
		if (lstFilePaths.size()>1)
			if (!ui.ckbLeave->isChecked())
				QFile::remove(strTempMergedFileOut);
	}
	else
	{
		//---------------------------------------------------------------
		// MERGE MPG FILES WITH MP4BOX COMMAND: mp4box -cat f1 -cat f2 target
		//---------------------------------------------------------------
		if (lstFilePaths.size()>1)
		{
			bSuccess = VideoHelper::MergeMp4(lstFilePaths,ui.lineEdit->text());
		}
	}

	this->statusBar()->showMessage("Deleting temporary files");
	m_ProgresBar->setValue(97);
	QCoreApplication::processEvents(QEventLoop::ExcludeUserInputEvents);

	//remove temporary files
	if (!ui.ckbLeave->isChecked())
	{
		for (int i=0;i<lstTempResized.size();i++)
		{
			QFile::remove(lstTempResized.at(i));
		}

		for (int i=0;i<lstTempResizedToMp4Fast.size();i++)
		{
			QFile::remove(lstTempResizedToMp4Fast.at(i));
		}

	}



	m_ProgresBar->setValue(100);
	if (bSuccess)
		QMessageBox::information(this,"Information","Operation success!");
}





/*
	Analyze all files, get audio and video data, compare if only one file is different then re-encode all files with same AV parameters and approx same frame size
*/
bool MergeMp4::AnaylizeFiles(QStringList lstFilePaths, bool &bNeed2Convert, QString &strFrameSize,QString &strFPS,QString &strDAR,int &nVideoBitRateMp4,int &nAudioBitRateMp4,int &nAudioBitRateMpg,int &nAudioFrequency,QStringList &lstVideoCodecs)
{
	bNeed2Convert=false; 

	// test all files if same size, if so then resize/convert and then merge */
	QStringList lstSizes;
	QList<int> lstBitrate;
	QStringList lstData;

	nAudioBitRateMp4=0;
	nAudioFrequency=0;
	int nAudioChannels=0;
	nAudioBitRateMpg=0;

	QString strAudioCodecCheck="";

	for (int i=0;i<lstFilePaths.size();i++)
	{
		QString strFile=lstFilePaths.at(i);
		QString strSize;
		int nVideoBitrate;
		QString strAuxData;
		int nFile_AudioBitrate;
		int nFile_AudioFrequency;
		int nFile_AudioChannels;
		QString strVideoCodec;
		QString strAudioCodec;
		

		if(!VideoHelper::GetSingleVideoFileInfo(strFile,strSize,nVideoBitrate,strAuxData,strFPS,strDAR,nFile_AudioBitrate,nFile_AudioFrequency,strVideoCodec,strAudioCodec,nFile_AudioChannels))
		{
			QMessageBox::warning(this,"Error",QString("File %1 can not be analyzed!").arg(strFile));
			return false;
		}

		lstSizes<<strSize;
		lstBitrate<<nVideoBitrate;
		lstData<<strAuxData;

		//check if video codec is same
		if (!lstVideoCodecs.contains(strVideoCodec) && lstVideoCodecs.size()>0)
		{
			bNeed2Convert=true; //different video codecs detected
		}
		lstVideoCodecs<<strVideoCodec;


		//check if audio codec is same
		if (strAudioCodecCheck.isEmpty())
			strAudioCodecCheck=strAudioCodec;
		else
		{
			if (strAudioCodec!=strAudioCodecCheck)
				bNeed2Convert=true; //different audio codecs detected
		}

		//set all to minimum (to avoid glitches):
		if (nAudioBitRateMp4>nFile_AudioBitrate || nAudioBitRateMp4==0)
			nAudioBitRateMp4=nFile_AudioBitrate;

		if (nAudioFrequency!=nFile_AudioFrequency)
		{
			if (nAudioFrequency!=0)
				bNeed2Convert=true; //if some file has different freq, mark all for re-encode

			if(nAudioFrequency>nFile_AudioFrequency || nAudioFrequency==0) //set freq to min
				nAudioFrequency=nFile_AudioFrequency;
		}

		if (nAudioChannels!=nFile_AudioChannels)
		{
			if (nAudioChannels!=0)
				bNeed2Convert=true; //if some file has different channels, mark all for re-encode
			if(nAudioChannels>nFile_AudioChannels  || nAudioChannels==0) //set channels to min
				nAudioChannels=nFile_AudioChannels;
		}

	}

	if (nAudioBitRateMp4==0)
		nAudioBitRateMp4=64;

	if (nAudioFrequency==0)
		nAudioFrequency=22500;

	if (strFPS.isEmpty())
	{
		QMessageBox::warning(this,"Error","Some files can not be open! Check if filenames contain special characters. If so, then rename them!");
		return false;
	}


	//audio stream will be converted to the mpeg1 format if all files are re-encoded to avoid out of audio sync problems
	//set audio bit rate 64, 96, 128 or 192, lowest mpg = 96 as mpg is lousy compression
	if (nAudioBitRateMp4<=48)
		nAudioBitRateMpg=48;
	else if (nAudioBitRateMp4<=96)
		nAudioBitRateMpg=96;
	else if (nAudioBitRateMp4<=128)
		nAudioBitRateMpg=128;
	else 
		nAudioBitRateMpg=160;

	//set mp4 audio bitrate
	//if (nAudioBitRateMp4<=48)
	//	nAudioBitRateMp4=48;
	if (nAudioBitRateMp4<64)
		nAudioBitRateMp4=64;
	else if (nAudioBitRateMp4<=96)
		nAudioBitRateMp4=96;
	else if (nAudioBitRateMp4<=128)
		nAudioBitRateMp4=128;
	else
		nAudioBitRateMp4=160;



	//get min video bitrate for h264 stream output:
	nVideoBitRateMp4=lstBitrate.at(0);
	for (int i=1;i<lstBitrate.size();i++)
	{
		if (nVideoBitRateMp4>lstBitrate.at(i))
			nVideoBitRateMp4=lstBitrate.at(i);
	}

	if (nVideoBitRateMp4>5000)//limit mp4 bitrate as sometimes parser can fail
	{
		nVideoBitRateMp4=400;
	}


	//test if video needs to be converted
	bool bNeed2ConvertVideo=false;
	strFrameSize = VideoHelper::GetMostCommonSize(lstSizes,lstData,bNeed2ConvertVideo);
	if (bNeed2ConvertVideo) bNeed2Convert=true;

	return true;
}



void MergeMp4::OnPathLinkActivated ( const QString &strUrl)
{
	QUrl urlLink = QUrl::fromLocalFile(strUrl);
	QDesktopServices::openUrl(urlLink);
}


